<?php
/******************************************************************************
 *
 * Subrion Real Estate Classifieds Software
 * Copyright (C) 2018 Intelliants, LLC <https://intelliants.com>
 *
 * This file is part of Subrion Real Estate Classifieds Software.
 *
 * This program is a commercial software and any kind of using it must agree
 * to the license, see <https://subrion.pro/license.html>.
 *
 * This copyright notice may not be removed from the software source without
 * the permission of Subrion respective owners.
 *
 *
 * @link https://subrion.pro/product/real-estate.html
 *
 ******************************************************************************/

class iaEstate extends abstractModuleFront implements iaRealEstateModule
{
    const RENT = 'rent';
    const SALE = 'sale';

    const SESSION_COMPARISON_KEY = 'realestate_comparison';
    const SESSION_SORTING_KEY = 'realestate_sorting';

    const COOKIE_GUEST_ACCESS_KEY = 'ga';

    protected static $_table = 'estates';

    protected $_itemName = 'estate';

    protected $_statuses = [
        self::STATUS_AVAILABLE,
        iaCore::STATUS_APPROVAL,
        self::STATUS_HIDDEN,
        self::STATUS_SOLD,
        self::STATUS_RENTED
    ];

    public $coreSearchEnabled = true;
    public $coreSearchOptions = [
        'tableAlias' => 'e',
        'columnAlias' => [
            'date' => 'date_modified',
            'year' => 'built_year'
        ],
        'regularSearchFields' => ['address'],
        'customColumns' => ['keyword', 'l', 'id', 'sl']
    ];

    private $_foundRows = 0;


    // Common system calls section
    public function getUrl(array $data)
    {
        return $this->getInfo('url')
            . 'view/'
            . (isset($data['url']) ? $data['url'] : '')
            . (isset($data[self::COLUMN_ID]) ? $data[self::COLUMN_ID] : '')
            . '.html';
    }

    public function url($action, array $listingData)
    {
        $patterns = [
            'account' => 'profile/properties/',
            'default' => ':base:action/:id/',
            'view' => ':base:action/:alias:id.html'
        ];

        $url = iaDb::printf(
            isset($patterns[$action]) ? $patterns[$action] : $patterns['default'],
            [
                'action' => $action,
                'alias' => isset($listingData['url']) ? $listingData['url'] : '',
                'base' => $this->getInfo('url'),
                'id' => isset($listingData[self::COLUMN_ID]) ? $listingData[self::COLUMN_ID] : ''
            ]
        );

        return $url;
    }

    public function accountActions($params)
    {
        return [$this->url(iaCore::ACTION_EDIT, $params['item']), null];
    }

    public function extraActions(array $params)
    {
        $output = (string)'';

        if ($params['member_id'] != iaUsers::getIdentity()->id) {
            return $output;
        }

        $htmlPattern = ' <a href=":href" class=" :class" title=":title"><span class="fa fa-:icon"></span></a>';

        $output .= iaDb::printf($htmlPattern, [
            'class' => 'js-removal-confirmation',
            'href' => $this->url(iaCore::ACTION_DELETE, $params),
            'icon' => 'times-circle',
            'title' => iaLanguage::get('delete')
        ]);

        $output .= iaDb::printf($htmlPattern, [
            'class' => '',
            'href' => $this->getInfo('url') . 'add/?' . $params['id'],
            'icon' => 'plus',
            'title' => iaLanguage::get('copy')
        ]);

        return $output;
    }

    // called at search pages
    public function coreSearch($stmt, $start, $limit, $order)
    {
        if ($order) {
            list($field, $direction) = explode(' ', $order);
            if (preg_match('#\`([a-z_]+?)\`#i', $field, $matches)) {
                $field = $matches[1];
            }
            $sorting = ['field' => $field, 'direction' => $direction];
        } else {
            $sorting = ['field' => 'date_modified', 'direction' => iaDb::ORDER_DESC];
        }

        $rows = $this->get($stmt, $start, $limit, $sorting);

        return [$this->getFoundRows(), $rows];
    }

    public function coreSearchTranslateColumn($column, $value)
    {
        $iaLocation = $this->iaCore->factoryModule('location', $this->getModuleName(), 'common');

        switch ($column) {
            case 'keyword':
                $value = iaSanitize::sql($value);
                $searchTerm = [];

                if (is_numeric($value)) {
                    return [
                        ['col' => ':column', 'cond' => '=', 'val' => $value, 'field' => 'zipcode'],
                        ['col' => ':column', 'cond' => '=', 'val' => $value, 'field' => 'id']
                    ];
                } elseif (2 == strlen($value) || 1 === preg_match('#([ \w]+), ([A-Z]{2})#i', $value, $searchTerm)) {
                    if ($searchTerm) {
                        $searchTerm = [
                            [
                                'column' => iaLocation::COLUMN_TITLE,
                                'cond' => iaLocation::SQL_CONDITION_EQUAL,
                                'value' => $searchTerm[1]
                            ],
                            [
                                'column' => iaLocation::FAKE_COLUMN_ABBREVIATION,
                                'cond' => iaLocation::SQL_CONDITION_EQUAL,
                                'value' => $searchTerm[2]
                            ]
                        ];
                    } else {
                        $searchTerm = [
                            [
                                'column' => iaLocation::COLUMN_CODE,
                                'cond' => iaLocation::SQL_CONDITION_EQUAL,
                                'value' => $value
                            ]
                        ];
                    }

                    $location = $iaLocation->search($searchTerm,
                        [iaLocation::COLUMN_ID, iaLocation::COLUMN_TITLE, iaLocation::FAKE_COLUMN_ABBREVIATION, iaLocation::COLUMN_URL],
                        ['field' => iaLocation::COLUMN_ID, 'direction' => iaDb::ORDER_ASC], 0, 1);

                    if ($location) {
                        $location = array_shift($location);

                        $stmt = '(';
                        if (isset($_GET['children_loc'])) {
                            if ($childrenList = $iaLocation->getChildren($location[iaLocation::COLUMN_ID])) {
                                foreach ($childrenList as $key => $item) {
                                    $stmt .= $item[iaLocation::COLUMN_ID] . iaLocation::VALUES_DELIMITER_CHAR;
                                }
                            }
                        }
                        $stmt .= $location[iaLocation::COLUMN_ID] . ')';

                        return ['col' => ':column', 'cond' => 'IN', 'val' => $stmt, 'field' => 'location_id'];
                    }
                } else {
                    $searchTerm = [
                        [
                            'column' => iaLocation::COLUMN_TITLE,
                            'cond' => iaLocation::SQL_CONDITION_LIKE,
                            'value' => $value
                        ]
                    ];

                    $locations = $iaLocation->search($searchTerm, [
                        iaLocation::COLUMN_ID,
                        iaLocation::COLUMN_TITLE,
                        iaLocation::NS_LEFT,
                        iaLocation::NS_RIGHT
                    ],
                        ['field' => iaLocation::COLUMN_ID, 'direction' => iaDb::ORDER_ASC], 0, 1);

                    if ($locations) {
                        //$this->iaView->assign('locations', $locations);

                        $where = [];
                        foreach ($locations as $entry) {
                            $sql = sprintf('(SELECT `nid` FROM `%s` WHERE `left` >= %d AND `right` <= %d)',
                                iaLocation::getTableTree(true), $entry['left'], $entry['right']);
                            $where[] = ['col' => ':column', 'cond' => 'IN', 'val' => $sql, 'field' => 'location_id'];
                        }

                        return $where;
                    } else {
                        return [
                            'col' => ':column',
                            'cond' => 'LIKE',
                            'val' => "'%" . $value . "%'",
                            'field' => 'address'
                        ];
                    }
                }

            case 'l':
                $node = $this->iaDb->row([iaLocation::NS_LEFT, iaLocation::NS_RIGHT], iaDb::convertIds($value, 'nid'),
                    $iaLocation::getTableTree());

                $stmt = sprintf('(SELECT `nid` FROM `%s` WHERE `left` >= %d AND `right` <= %d)',
                    $iaLocation::getTableTree(true),
                    $node[iaLocation::NS_LEFT],
                    $node[iaLocation::NS_RIGHT]);

                return ['col' => ':column', 'cond' => 'IN', 'val' => $stmt, 'field' => 'location_id'];

            case 'sl':
                return ['col' => ':column', 'cond' => '=', 'val' => (int)$value, 'field' => 'location_id'];
        }
    }

    // called at the Member Details page
    public function fetchMemberListings($memberId, $start, $limit)
    {
        return [
            'items' => $this->get(['member_id' => $memberId], $start, $limit,
                ['field' => 'date_modified', 'direction' => iaDb::ORDER_DESC]),
            'total_number' => $this->iaDb->foundRows()
        ];
    }

    /**
     * Returns listings for Favorites page
     *
     * @param $ids
     * @param $fields
     *
     * @return mixed
     */
    public function getFavorites($ids, $fields)
    {
        $rows = $this->get('e.`id` IN (' . implode(',', $ids) . ')', 0, 100,
            ['field' => 'date_modified', 'direction' => iaDb::ORDER_DESC]);

        if ($rows) {
            foreach ($rows as &$row) {
                $row['favorite'] = 1;
            }
        }

        return $rows;
    }

    // overloads predefined method
    public function getById($id, $ignoreStatus = false, $decorate = true)
    {
        if ($listing = $this->get([self::COLUMN_ID => (int)$id], 0, 1, [
            'field' => iaDb::ID_COLUMN_SELECTION,
            'direction' => iaDb::ORDER_DESC
        ], $ignoreStatus, $decorate)
        ) {
            $listing = array_shift($listing);
            $listing['country'] = $this->iaCore->get('realestate_country_name');
        }

        return $listing;
    }

    public function postPayment(array $plan, array $transaction)
    {
        $this->iaDb->update(['status' => self::STATUS_AVAILABLE], iaDb::convertIds($transaction['item_id']), null,
            self::getTable());
    }

    public function get($conditions, $start, $limit, $order = ['field' => self::COLUMN_ID, 'direction' => 'ASC'], $ignoreStatuses = false, $decorate = true)
    {
        $sqlOrder = 'e.`sponsored` DESC, ';
        $sqlWhere = iaDb::EMPTY_CONDITION;

        if ($conditions) {
            if (is_array($conditions)) {
                foreach ($conditions as $key => $value) {
                    $sqlWhere .= iaDb::printf(" AND e.`:column` = ':value'",
                        ['column' => $key, 'value' => iaSanitize::sql($value)]);
                }
            } else {
                $sqlWhere = $conditions;
            }
        }

        if (!$ignoreStatuses) {
            $sqlWhere .= " AND m.`status` = '" . iaCore::STATUS_ACTIVE . "' AND e.`status` IN ('"
                . implode("','", $this->_getDisplayableStatuses()) . "')";
        }

        is_array($order)
            ? $sqlOrder .= iaDb::printf('e.`:field` :direction', $order)
            : $sqlOrder = $order;

        $this->iaCore->factoryModule('location', $this->getModuleName(), 'common');

        $sql = <<<SQL
SELECT SQL_CALC_FOUND_ROWS e.*, l.`url`, l.`title_:lang` `location`, l2.`title_:lang` `state`, t.`parent` `l`, 
  CONCAT(l.`title_:lang`, IF(l2.`title_:lang` != "", CONCAT(", ", l2.`title_:lang`), ""), " ", e.`zipcode`) `address2` 
FROM `:table` e 
LEFT JOIN `:locations` l ON (l.`id` = e.`location_id`) 
LEFT JOIN `:locations_tree` t ON (t.`nid` = e.`location_id`) 
LEFT JOIN `:locations` l2 ON (l2.`id` = t.`parent`) 
LEFT JOIN `:members` m ON (m.`id` = e.`member_id`) 
WHERE :conditions 
ORDER BY :order 
LIMIT :start, :limit
SQL;
        $sql = iaDb::printf($sql, [
            'lang' => $this->iaCore->language['iso'],
            'table' => self::getTable(true),
            'locations' => iaLocation::getTable(true),
            'locations_tree' => iaLocation::getTableTree(true),
            'members' => iaUsers::getTable(true),
            'conditions' => $sqlWhere,
            'order' => $sqlOrder,
            'start' => $start,
            'limit' => $limit
        ]);

        $result = $this->iaDb->getAll($sql);
        $this->_foundRows = $this->iaDb->foundRows();

        $decorate && $this->_processValues($result);

        return $result;
    }

    public function getFoundRows()
    {
        return $this->_foundRows;
    }

    public function insert(array $entryData)
    {
        $entryData[self::COLUMN_ID] = $this->_generateId();
        $entryData['member_id'] = iaUsers::hasIdentity() ? iaUsers::getIdentity()->id : 0;
        $entryData['date_added'] = date(iaDb::DATETIME_FORMAT);
        $entryData['date_modified'] = date(iaDb::DATETIME_FORMAT);

        $this->iaDb->insert($entryData, null, self::getTable());

        $result = (0 == $this->iaDb->getErrorNumber()) ? $entryData[self::COLUMN_ID] : false;

        if ($result) {
            $this->updateCounters($entryData[self::COLUMN_ID], $entryData, iaCore::ACTION_ADD);

            // finally, notify plugins
            $this->iaCore->startHook('phpListingAdded', [
                'itemId' => $entryData[self::COLUMN_ID],
                'itemName' => $this->getItemName(),
                'itemData' => $entryData
            ]);

            iaUsers::hasIdentity() || $this->_rememberUsersListing($entryData);
        }

        return $result;
    }

    public function update(array $entryData, $id)
    {
        $entryData['date_modified'] = date(iaDb::DATETIME_FORMAT);

        return parent::update($entryData, $id);
    }

    public function updateCounters($itemId, array $itemData, $action, $previousData = null)
    {
        $this->iaCore->factoryModule('location', $this->getModuleName(), 'common')
            ->editCounters($action, $itemData, $previousData);

        if (iaCore::ACTION_DELETE == $action) {
            // remove from the Recently Viewed list
            if (isset($_SESSION[self::SESSION_RECENT_KEY][$itemId])) {
                unset($_SESSION[self::SESSION_RECENT_KEY][$itemId]);
            }
            //

            // finally, remove from comparing list if exists
            if (isset($_SESSION[self::SESSION_COMPARISON_KEY][$itemId])) {
                unset($_SESSION[self::SESSION_COMPARISON_KEY][$itemId]);
            }
        }
    }

    public function sortingParams(array $keys, array $defaults)
    {
        $result = [
            'direction' => 'DESC',
            'field' => 'date_modified'
        ];

        if (isset($defaults[self::SESSION_SORTING_KEY])) {
            $result = $defaults[self::SESSION_SORTING_KEY];
        }
        if (isset($keys['sort'])) {
            $validSortingFields = ['date_modified' => 'date', 'views_num' => 'views', 'price', 'built_year' => 'year'];
            if (in_array($keys['sort'], $validSortingFields)) {
                $key = array_search($keys['sort'], $validSortingFields);
                $result['field'] = is_numeric($key) ? $keys['sort'] : $key;
            }
        }
        if (isset($keys['order']) && in_array(strtolower($keys['order']), ['asc', 'desc'])) {
            $result['direction'] = strtoupper($keys['order']);
        }

        return $result;
    }

    protected function _processValues(&$rows, $singleRow = false, $fieldNames = [])
    {
        parent::_processValues($rows);

        if ($rows) {
            $singleRow && $rows = [$rows];

            foreach ($rows as &$row) {
                $row['gallery_num'] = count($row['gallery']);
                $row['size_formatted'] = number_format($row['size']);
            }

            $singleRow && $rows = array_shift($rows);
        }
    }

    protected function _getDisplayableStatuses()
    {
        $result = [self::STATUS_AVAILABLE, self::STATUS_RENTED];

        if ($this->iaCore->get('realestate_display_sold', true)) {
            $result[] = self::STATUS_SOLD;
        }

        return $result;
    }

    protected function _generateId()
    {
        $iaEstateHelper = $this->iaCore->factoryModule('helper', $this->getModuleName(), 'common');

        $iterationCounter = 1;
        $id = $iaEstateHelper->generateId();

        while ($this->iaDb->exists(iaDb::convertIds($id, self::COLUMN_ID), null, self::getTable())) {
            $iterationCounter++;
            if ($iterationCounter > 10) {
                return false;
            }
            $id = $iaEstateHelper->generateId();
        }

        return $id;
    }

    protected function _rememberUsersListing(array $listingData)
    {
        $ids = isset($_COOKIE[self::COOKIE_GUEST_ACCESS_KEY]) ? explode(',',
            $_COOKIE[self::COOKIE_GUEST_ACCESS_KEY]) : [];
        $ids[] = $this->createUserAccessHash($listingData);
        $ids = implode(',', $ids);

        $lifetime = 60 * 60 * 24 * 365 * 2; // 2 years

        setcookie(self::COOKIE_GUEST_ACCESS_KEY, $ids, time() + $lifetime, '/');
    }

    public function createUserAccessHash(array $listingData)
    {
        return md5($listingData['date_added'] . IA_SALT . $listingData['id']);
    }
}
