<?php
/******************************************************************************
 *
 * Subrion Real Estate Classifieds Software
 * Copyright (C) 2018 Intelliants, LLC <https://intelliants.com>
 *
 * This file is part of Subrion Real Estate Classifieds Software.
 *
 * This program is a commercial software and any kind of using it must agree
 * to the license, see <https://subrion.pro/license.html>.
 *
 * This copyright notice may not be removed from the software source without
 * the permission of Subrion respective owners.
 *
 *
 * @link https://subrion.pro/product/real-estate.html
 *
 ******************************************************************************/

class iaBackendController extends iaAbstractControllerModuleBackend
{
    protected $_name = 'properties';

    protected $_helperName = 'estate';

    protected $_gridFilters = [
        'address' => self::LIKE,
        'type' => self::EQUAL,
        'id' => self::EQUAL,
        'category' => self::EQUAL,
        'location' => self::EQUAL,
        'location_id' => self::EQUAL,
        'status' => self::EQUAL
    ];
    protected $_gridSorting = ['location' => ['title', 'l']];
    protected $_gridQueryMainTableAlias = 'e';

    protected $_activityLog = ['item' => 'property'];

    private $_iaLocation;


    public function __construct()
    {
        parent::__construct();

        $this->_iaLocation = $this->_iaCore->factoryModule('location', $this->getModuleName(), 'common');
    }

    protected function _indexPage(&$iaView)
    {
        parent::_indexPage($iaView);

        // connect the fancybox lib
        $iaView->add_css('_IA_URL_modules/fancybox/js/jquery.fancybox');
        $iaView->add_js('_IA_URL_modules/fancybox/js/jquery.fancybox.pack');
        //
    }

    protected function _setPageTitle(&$iaView, array $entryData, $action)
    {
        $iaView->title(iaLanguage::get($action . '_listing', $iaView->title()));
    }

    protected function _gridModifyOutput(array &$entries)
    {
        foreach ($entries as &$entry) {
            $entry['type'] = iaField::getLanguageValue($this->getItemName(), 'type', $entry['type']);

            $images = [];
            if ($gallery = unserialize($entry['gallery'])) {
                foreach ($gallery as $image) {
                    // pass the large image
                    $images[] = [
                        'href' => $this->_iaCore->iaView->assetsUrl . 'uploads/' . $image['path'] . 'large/' .  $image['file'],
                        'title' => $image['title']
                    ];
                }
            }

            $entry['gallery'] = $images;
        }
    }

    protected function _gridQuery($columns, $where, $order, $start, $limit)
    {
        $sql = <<<SQL
SELECT e.id, e.type, CONCAT(e.address, ", ", l.title_:lang) address, e.gallery, e.date_added,
    e.`status`, 1 `update`, 1 `delete`, 1 `copy`
FROM `:prefix:table_estates` e 
LEFT JOIN `:prefix:table_locations` l ON (e.location_id = l.id) 
WHERE :where :order 
LIMIT :start, :limit
SQL;
        $sql = iaDb::printf($sql, [
            'lang' => $this->_iaCore->language['iso'],
            'prefix' => $this->_iaDb->prefix,
            'table_estates' => $this->getTable(),
            'table_locations' => iaLocation::getTable(),
            'columns' => $columns,
            'where' => $where,
            'order' => $order,
            'start' => $start,
            'limit' => $limit
        ]);

        return $this->_iaDb->getAll($sql);
    }

    protected function _insert(array $entryData)
    {
        $entryData['id'] = $this->_getNewEntryId();
        $entryData['date_added'] = date(iaDb::DATETIME_FORMAT);
        $entryData['date_modified'] = date(iaDb::DATETIME_FORMAT);

        $this->getHelper()->insert($entryData);

        return (0 === $this->_iaDb->getErrorNumber()) ? $entryData['id'] : false;
    }

    protected function _update(array $entryData, $entryId)
    {
        $entryData['date_modified'] = date(iaDb::DATETIME_FORMAT);

        return parent::_update($entryData, $entryId);
    }

    public function updateCounters($entryId, array $entryData, $action, $previousData = null)
    {
        $this->_iaLocation->editCounters($action, $entryData, $previousData);

        if (iaCore::ACTION_DELETE) {
            // remove from the Recently Viewed list
            if (isset($_SESSION[iaEstate::SESSION_RECENT_KEY][$entryId])) {
                unset($_SESSION[iaEstate::SESSION_RECENT_KEY][$entryId]);
            }
        }
    }

    protected function _setDefaultValues(array &$entry)
    {
        $entry = [
            'member_id' => iaUsers::getIdentity()->id,
            'location_id' => 0,
            'date_added' => date(iaDb::DATETIME_SHORT_FORMAT),
            'featured' => false,
            'sponsored' => false,
            'status' => iaEstate::STATUS_AVAILABLE
        ];

        if (1 == count($_GET)) {
            $key = key($_GET);
            if (is_numeric($key) && !$_GET[$key]) {
                if ($row = $this->getById($key)) {
                    $entry = $row;
                    $entry['status'] = iaEstate::STATUS_AVAILABLE;
                    unset($entry[iaDb::ID_COLUMN_SELECTION]);
                }
            }
        }
    }

    protected function _preSaveEntry(array &$entry, array $data, $action)
    {
        parent::_preSaveEntry($entry, $data, $action);

        $entry['location_id'] = (int)$data['location'];
        if (empty($entry['location_id'])) {
            $this->addMessage('location_is_not_set');
        } else {
            $categoryData = $this->_iaLocation->getById($entry['location_id']);
            if (empty($categoryData)) {
                $this->addMessage('chosen_location_is_incorrect');
            }
        }

        return !$this->getMessages();
    }

    protected function _assignValues(&$iaView, array &$entryData)
    {
        parent::_assignValues($iaView, $entryData);

        $entryData['parents'] = $this->_iaLocation->getParentsFlat($entryData['location_id']);

        $iaView->assign('categories', $this->_iaLocation->getChildrenList([]));
        $iaView->assign('statuses', $this->getHelper()->getStatuses());
    }

    protected function _getJsonTypes()
    {
        $data = [];

        if ($fieldData = $this->_iaDb->getEnumValues($this->getTable(), 'type')) {
            foreach ($fieldData['values'] as $type) {
                $data[] = ['value' => $type, 'title' => iaLanguage::get('field_estate_type+' . $type)];
            }
        }

        return ['data' => $data];
    }

    private function _getNewEntryId()
    {
        $iaEstateHelper = $this->_iaCore->factoryModule('helper', $this->getModuleName(), 'common');

        $iterationCounter = 1;
        $listingId = $iaEstateHelper->generateId();

        while ($this->_iaDb->exists(iaDb::convertIds($listingId))) {
            $iterationCounter++;
            if ($iterationCounter > 10) {
                return false;
            }

            $listingId = $iaEstateHelper->generateId();
        }

        return $listingId;
    }

    protected function _writeLog($action, array $entryData, $entryId)
    {
        $iaLog = $this->_iaCore->factory('log');

        $actionsMap = [
            iaCore::ACTION_ADD => iaLog::ACTION_CREATE,
            iaCore::ACTION_EDIT => iaLog::ACTION_UPDATE,
            iaCore::ACTION_DELETE => iaLog::ACTION_DELETE
        ];

        $title = '#' . $entryId;

        $iaLog->write($actionsMap[$action], array_merge($this->_activityLog, ['name' => $title, 'id' => $entryId]));
    }
}
